# Copyright (c) 2010 - 2023 VMware, Inc. All rights reserved.
# VMware Confidential

import os
import logging, logging.handlers, logging.config
from serviceconfig import ServiceConfig, PLAT_IND_CIS_HOME, ConfigError, \
        _scm_load_properties
import xml.etree.ElementTree as ET
import socket
import subprocess
import json
import sys
import time
import zlib

XML_SETUP = '/etc/vmware-rbd/autodeploy-setup.xml'

LOG_SIZE = 10 * 1024 * 1024
LOG_ROTATE_COUNT = 5
SOLUSER_NAME = 'vpxd-extension'

FIREWALL_CONFIG='/etc/vmware/appliance/firewall/vmware-autodeploy'
FIREWALL_SCRIPT='/usr/lib/applmgmt/networking/bin/firewall-reload'

PORT_IN_USE = 'cis.error.portinuse'
INVALID_ARGUMENT = 'cis.error.invalidValue'

# XXX: The following methods and the LOG_FORMAT mirror what's in logutil.py.
# Make sure they remain in sync. It will be nice to actually move them to a
# separate module that can be shared between frozen binaries and stand-alone
# script.
LOG_FORMAT = "%(asctime)s [%(process)d]%(levelname)s:%(module)s:%(message)s"

class vmwFormatter(logging.Formatter):
    def formatTime(self, record, datefmt=None, msecFmt="%s.%03d"):
        ct = self.converter(record.created)
        if datefmt:
            s = time.strftime(datefmt, ct)
        else:
            t = time.strftime("%Y-%m-%dT%H:%M:%S", ct)
            s = msecFmt % (t, record.msecs)
        return s

def changeLogFileOwner(logPath, user='deploy'):
    import pwd
    try:
        if pwd.getpwuid(os.stat(logPath).st_uid).pw_name != user:
            try:
                pw = pwd.getpwnam(user)
                current_umask = os.umask(0)
                os.chown(logPath, pw.pw_uid, pw.pw_gid)
            except:
                logging.exception('Error while changing ownership: name: %s'
                                  % logPath)
                raise
    except:
        logging.exception('Unable to find the owner for %s' % logPath)
        raise

def setupLogger(logDir):
    rootLogger = logging.getLogger()
    if logDir:
        logPath = os.path.join(logDir, 'rbd-sca.log')
        if os.path.exists(logPath):
            changeLogFileOwner(logPath)
        handler = logging.handlers.RotatingFileHandler(
                            filename = logPath,
                            mode = "a",
                            maxBytes = LOG_SIZE,
                            backupCount = LOG_ROTATE_COUNT)
        handler.rotator = rotator
        handler.namer = namer
        handler.setLevel(logging.DEBUG)
        handler.setFormatter(vmwFormatter(LOG_FORMAT))
        rootLogger.addHandler(handler)
        rootLogger.setLevel(logging.DEBUG)

def namer(name):
    return name + ".gz"

def rotator(source, dest):
    with open(source, "rb") as sf:
        data = sf.read()
        compressed = zlib.compress(data, 9)
        with open(dest, "wb") as df:
            df.write(compressed)
    os.remove(source)

class AutoDeployConfig(ServiceConfig):
    '''
    The get function gets invoked when a refresh happens
    This function invokes notify which will basically
    go and update the firewall changes. While the original
    intention of notify was basically during changes, we are
    using this which will help us in backup/restore.
    Although now a get operation is a bit expensive, its
    only invoked when someone lands on the settings page and
    refreshes it, so it is not a function which gets invoked
    all the time.
    '''
    def _get(self):
        self._notify()
        super(AutoDeployConfig, self)._get()

    '''
    Perform the validations
    for various configuration
    provided by autodeploy
    For instance if the port is in use
    Throw Config Error
    saying the port cannot be modified
    to mentioned one
    '''
    def _validate(self):
        '''
        Grab the key value
        '''
        logging.info('Validating the parameters for autodeploy')
        try:
            props = _scm_load_properties(sys.stdin.buffer)
        except:
            props = _scm_load_properties(sys.stdin)
        '''
        Check if we are able to bind to the ports
        '''
        for key in props:
            if key == 'serviceport' or key == 'managementport':
                if not props[key].isdigit():
                    ConfigError(INVALID_ARGUMENT, key,
                            'Please provide integer value for port').fatal()
                else:
                    port = int(props[key])
                    if port > 1024 and port <=65535:
                        sock = socket.socket(socket.AF_INET6,
                                socket.SOCK_STREAM)
                        try:
                            sock.bind(('', port))
                        except Exception as e:
                            ConfigError(PORT_IN_USE, key,
                                    'The provided port value is already in use').fatal()
                        finally:
                            sock.close()
                    elif port >=0 and port <=1024:
                        ConfigError(INVALID_ARGUMENT, key,
                                    'Provided port is a privileged port').fatal()
                    else:
                        ConfigError(INVALID_ARGUMENT, key,
                            'Please provide a port number in valid range').fatal()

            if key == 'loglevel':
                debugLevel = props[key]
                if debugLevel not in ['INFO', 'DEBUG', 'ERROR', 'WARNING','CRITICAL']:
                    ConfigError(INVALID_ARGUMENT, key,
                                      'The provided loglevel is not supported.'
                                      ' Please enter values INFO, DEBUG, ERROR,'
                                      ' WARNING, CRITICAL').fatal()

            if key == 'cachesize_GB':
                cachesize = props[key]
                # As far cache size verification is concerned
                # its best we let autodeploy figure out and throw
                # appropriate error
                if not cachesize.isdigit() or int(cachesize)<=0:
                    ConfigError(INVALID_ARGUMENT, key,
                            'Please provide a integer value size in GiB').fatal()

                # The backup temp file might potentially be placed in the same
                # partition as the cache. So enforce the rule that the cache
                # size cannot be more than half of the partition size.

                # Calculate disk size in GB.
                diskStat = os.statvfs('/var/lib/rbd/cache')
                diskSize = ((diskStat.f_bavail * diskStat.f_frsize)
                             // (1024 * 1024 * 1024))

                if int(cachesize) > diskSize // 2:
                    ConfigError(INVALID_ARGUMENT, key,
                        'cachesize_GB(%s) cannot be more than half of '
                        'partition size(%s)' % (cachesize,
                                                diskSize)).fatal()
        logging.info('Successfully validated the parameters')

    def _updateHealthUrl(self, managementPort):
        '''
        Updates the Health URL in LS
        '''
        sys.path.append(os.environ['VMWARE_PYTHON_PATH'])
        from cis.cisreglib import do_lsauthz_operation, parse_cisreg_prop
        from cis.defaults import get_cis_install_dir, get_component_home_dir
        from cis.utils import FileBuffer
        from cis.vecs import vmafd_machine_id
        CMBin = os.path.join(get_cis_install_dir(),
                             get_component_home_dir("cm"),
                             "bin")
        sys.path.append(CMBin)
        from cloudvmcisreg import read_and_substitute_options_file, SCA_SERVICES_DIR

        soluserName = 'vpxd-extension-%s' % vmafd_machine_id()

        specProps = {
          'control.script' : 'autodeploy-sca.sh',
          'autodeploy.ext.managementport' : managementPort,
          'solution-user.name' : soluserName,
          'unsupported.action' : 'SET_STARTUPTYPE_DISABLED',
        }

        fbPropsFile = os.path.join('/etc/vmware-rbd', 'firstboot',
                                  'autodeploy.properties')


        propsBuffer = FileBuffer()
        propsBuffer.readFile(fbPropsFile)
        read_and_substitute_options_file(propsBuffer, dynVars = specProps)
        urlMap = {}
        urls = propsBuffer.getBufferContentsByPattern('url')
        for url in urls:
            keyVal = url.split('=', 1)
            if len(keyVal) == 2:
                urlMap[keyVal[0].strip()] = keyVal[1].strip()

        logging.info('The new urls: %s' % urlMap)
        cisreg_opts_dict = parse_cisreg_prop(propsBuffer.getBufferContents())
        cisreg_opts_dict['cisreg.op'] = 'reregister'
        cisreg_opts_dict['sso.vecs.name'] = cisreg_opts_dict['sso.key.alias'] = SOLUSER_NAME
        serviceId = do_lsauthz_operation(cisreg_opts_dict)

        # Update the sca spec file
        outputBuffer = FileBuffer()
        outputBuffer.readFile(os.path.join(SCA_SERVICES_DIR, 'rbd.properties'))
        outputBuffer.updateKeyValue('health.location', urlMap['health.url'])
        outputBuffer.updateKeyValue('resourcebundle.location', urlMap['resourcebundle.url'])
        outputBuffer.writeFile(os.path.join(SCA_SERVICES_DIR, 'rbd.properties'))

    def _notify(self):
        '''
        When the parameter is being set
        we have to basically modify the XML
        so that when the user performs a
        restart those values get updated
        '''
        try:
            props, attrs = self._load_all()
            firewallInfo = []
            mgmtPortChanged = False

            xml = ET.parse(XML_SETUP)
            defaultValues = xml.getroot().find('defaultValues')

            if defaultValues != None:
                for child in defaultValues:
                    if child.tag == 'port':
                        if child.text != props['serviceport']:
                            firewallInfo.append(['delete', child.text])
                            firewallInfo.append(['add',  props['serviceport']])
                        child.text = props['serviceport']

                    elif child.tag == 'portAdd':
                        if child.text != props['managementport']:
                            firewallInfo.append(['delete', child.text])
                            firewallInfo.append(['add', props['managementport']])
                            mgmtPortChanged = True
                        child.text = props['managementport']

                    elif child.tag == 'maxSize':
                        child.text = props['cachesize_GB']

            debug = xml.getroot().find('debug')
            if debug != None:
                for child in debug:
                    if child.tag == 'level':
                        child.text = props['loglevel']

            with open(XML_SETUP, 'wb') as fp:
                # ET.tostring produces encoded bytes.
                fp.write(ET.tostring(xml.getroot(), encoding='utf-8'))

            #Perform firewall changes
            with open(FIREWALL_CONFIG, 'rb') as fp:
                data = fp.read().decode('utf-8')
                firewallData = json.loads(data)
                '''
                Expecting file will be in this format,
                else we want exception to be thrown for key error
                '''
                for rule in firewallData['firewall']['rules']:
                    '''
                    In the firewall config, rule name is
                    'autodeploy.ext.xyz'. Autodeploy's config has just
                    'xyz'. So trim the name got from the firewall config.
                    '''
                    ruleName = rule['name'].lower().split('.')[-1]
                    if ruleName in props:
                        rule['port'] = props[ruleName]
                        logging.info('Firewall data port %s ' % rule['port'])

            with open(FIREWALL_CONFIG, 'wb') as fp:
                # json.dumps returns bytes on python2 and str on python3.
                fwData = json.dumps(firewallData, indent=4, sort_keys=True)
                if not isinstance(fwData, bytes):
                    fwData = fwData.encode('ascii')
                fp.write(fwData)

            obj = subprocess.Popen(FIREWALL_SCRIPT, stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE)
            stdout, stderr = obj.communicate()

            logging.info('Reconfigured the firewall service stdout:%s,'
                    ' stderr:%s, returncode:%s' % (stdout, stderr,
                        obj.returncode))
            if mgmtPortChanged:
                self._updateHealthUrl(props['managementport'])
                logging.info('Updated the health url to point to %s',
                            props['managementport'])

        except Exception as e:
            logging.exception('Something went wrong while performing an update')

configPath = '%srbd' % PLAT_IND_CIS_HOME
logPath = os.path.join(os.environ['VMWARE_LOG_DIR'], 'vmware', 'rbd')

propsPath = os.path.join(configPath, 'config', 'autodeploy-config.props')
attrPath = os.path.join(configPath, 'config', 'autodeploy-config.attrs')
'''
If anything goes wrong , it should not be marked as fatal
This is just for some debugging purposes
'''
try:
    setupLogger(logPath)
except:
    pass

AutoDeployConfig(propsPath, attrPath)
