__copyright__ = 'VMware, Inc 2016'
'''Update Manager SCA config script'''
# https://opengrok.eng.vmware.com/source/xref/vsphere-2015.perforce.1666/bora/cis/services/sca/src/main/resources/serviceconfig.py

import os
import sys
import logging
from serviceconfig import ServiceConfig, PLAT_IND_CIS_HOME, ConfigError, \
    _scm_load_properties
import socket
import json
import subprocess
import xml.etree.ElementTree as ET

#__python2_hack__
try:
    from urllib.parse import urlparse
except ImportError:
    from urlparse import urlparse

#__python2_hack__
try:
    SOURCE = sys.stdin.buffer
except AttributeError:
    SOURCE = sys.stdin

PORT_IN_USE = 'cis.error.portinuse'
INVALID_ARGUMENT = 'cis.error.invalidValue'

FIREWALL_CONFIG = '/etc/vmware/appliance/firewall/vmware-updatemgr'
FIREWALL_SCRIPT = '/usr/lib/applmgmt/networking/bin/firewall-reload'
UPDATEMGR_CONFIG_XML = '/usr/lib/vmware-updatemgr/bin/vci-integrity.xml'
UPDATEMGR_EXTENSION_XML = '/usr/lib/vmware-updatemgr/bin/extension.xml'

PROPS_PREFIX = 'com.vmware.vcIntegrity.scm.'
SOAP_PORT = 'soapPort'
WEB_PORT = 'webServerPort'
WEB_SSL_PORT = 'webSslPort'
LOG_LEVEL = 'loglevel'
UPDATEMGR_PORTS_PREFIX = 'updatemgr.ext.'

class UpdateManagerConfig(ServiceConfig):
    def _validate(self):
        '''
        Validate modifications to updatemgr configuration parameters
        '''
        logging.info('Validating the parameters for updatemgr')
        props = _scm_load_properties(SOURCE)
        logLevels = ['VERBOSE', 'INFO', 'WARNING', 'ERROR', 'PANIC', 'QUIET']
        for key in props:
            if key in [PROPS_PREFIX + suffix for suffix in [SOAP_PORT, WEB_PORT, WEB_SSL_PORT]]:
                if not props[key].isdigit():
                    ConfigError(INVALID_ARGUMENT, key,
                            'Please provide integer value for port').fatal()
                else:
                    port = int(props[key])
                    if port >= 1024 and port <= 65535:
                        sock = socket.socket(socket.AF_INET6,
                                socket.SOCK_STREAM)
                        try:
                            sock.bind(('', port))
                        except Exception as e:
                            ConfigError(PORT_IN_USE, key,
                            'The provided port value is already in use').fatal()
                    else:
                        ConfigError(INVALID_ARGUMENT, key,
                        'Please provide a port number in valid range. (1024 - 65535)').fatal()

            if key == PROPS_PREFIX + LOG_LEVEL:
                debugLevel = props[key]
                if debugLevel not in logLevels:
                    ConfigError(INVALID_ARGUMENT, key,
                        'The provided loglevel is not supported.'
                        ' Please enter values VERBOSE, INFO, WARNING, ERROR,'
                        ' PANIC, QUIET').fatal()

        logging.info('Successfully validated the parameters')

    def _reconfig_firewall(self, props):
        ''' Reconfiguring the firewall with new values of props
        '''

        with open(FIREWALL_CONFIG, 'r') as fp:
            firewallData = json.load(fp)
            # Expecting file will be in this format,
            # else we want exception to be thrown for key error
            for rule in firewallData['firewall']['rules']:
                portShort = rule['name'].replace(UPDATEMGR_PORTS_PREFIX, '')
                if PROPS_PREFIX + portShort in props:
                    rule['port'] = props[PROPS_PREFIX + portShort]
                    logging.info('Firewall data port %s ' % rule['port'])

        with open(FIREWALL_CONFIG, 'w') as fp:
            fp.write(json.dumps(firewallData, indent=4, sort_keys=True))

        obj = subprocess.Popen(FIREWALL_SCRIPT, stdout=subprocess.PIPE,
                              stderr=subprocess.PIPE)
        stdout, stderr = obj.communicate()

        logging.info('Reconfigured the firewall service stdout:%s,'
                ' stderr:%s, returncode:%s' % (stdout, stderr,
                    obj.returncode))

    def _updateConfigXml(self, props):
        ''' Updating vci-integrity.xml with new values of props
        '''
        xml = ET.parse(UPDATEMGR_CONFIG_XML)
        root = xml.getroot()

        # Replace ufa_agent ports.
        for prop in [SOAP_PORT, WEB_PORT, WEB_SSL_PORT]:
            elem = root.find('./plugins/ufa_agent/' + prop)
            if elem is not None:
                elem.text = props[PROPS_PREFIX + prop]

        # Update the loglevel
        loglevelElem = root.find('./log/level')
        if loglevelElem is not None:
            loglevelElem.text = props[PROPS_PREFIX + LOG_LEVEL].lower()

        # write the updated vci-integrity.xml file
        with open(UPDATEMGR_CONFIG_XML, 'wb') as fp:
            fp.write(ET.tostring(root))

    def _updatePortInUrl(self, url, newPortStr):
        ''' Update the existing port in a given url with new port
            Returns updated url
        '''
        parseResult = urlparse(url)
        newUrl = '{}://{}:{}{}'.format(parseResult.scheme,
            parseResult.hostname, newPortStr, parseResult.path)
        return newUrl

    def _updateExtensionXml(self, props):
        ''' Updating extension.xml with new values of props
        '''
        xml = ET.parse(UPDATEMGR_EXTENSION_XML)
        root = xml.getroot()

        # Update the healthUrl
        healthUrl = root.find('./extension/healthUrl')
        healthUrl.text = self._updatePortInUrl(healthUrl.text,
            props[PROPS_PREFIX + WEB_PORT])

        # Update the server locations
        for server in root.iter('server'):
            serverUrl = server.find('url')
            serverType = server.find('type')
            if serverType.text == 'SOAP':
                serverUrl.text = self._updatePortInUrl(serverUrl.text,
                    props[PROPS_PREFIX + SOAP_PORT])
            elif serverType.text == 'HTTP':
                serverUrl.text = self._updatePortInUrl(serverUrl.text,
                    props[PROPS_PREFIX + WEB_SSL_PORT])

        # Update the client locations
        for client in root.iter('client'):
            clientUrl = client.find('url')
            clientType = client.find('type')
            if clientType.text == 'win32':
                clientUrl.text = self._updatePortInUrl(clientUrl.text,
                    props[PROPS_PREFIX + WEB_PORT])

        # write the updated vci-integrity.xml file
        with open(UPDATEMGR_EXTENSION_XML, 'wb') as fp:
            fp.write(ET.tostring(root))

    def _notify(self):
        ''' Called after the new values are set to services can act
            accordingly and read the new values
        '''
        logging.info('Reloading the firewall with correct values')
        props, attrs = self._load_all()
        # Reconfigure the firewall so that correct ports are open
        # Updatemgr service if running is unusable at this point until
        # it is restarted
        self._reconfig_firewall(props)

        # Update the ports in vci-integrity.xml
        self._updateConfigXml(props)

        # Update the ports in extension.xml
        self._updateExtensionXml(props)

configPath = '%supdatemgr' % PLAT_IND_CIS_HOME
propsPath = os.path.join(configPath, 'config', 'updatemgr-config.props')
attrPath = os.path.join(configPath, 'config', 'updatemgr-config.attrs')

UpdateManagerConfig(propsPath, attrPath)
