#!/usr/bin/env python

# Copyright (c) 2021-2022 VMware, Inc. All rights reserved.
# VMware Confidential

# This file goes in /usr/lib/vmware-sca/scripts/configuration

import os
import logging, logging.handlers, logging.config
from serviceconfig import ServiceConfig, PLAT_IND_CIS_HOME, ConfigError, \
        _scm_load_properties
import builtins
import socket
import subprocess
import json
import sys

LOG_FORMAT = "%(asctime)s [%(process)d]%(levelname)s:%(module)s:%(message)s"

LOG_SIZE = 10 * 1024 * 1024
LOG_ROTATE_COUNT = 5

FIREWALL_CONFIG='/etc/vmware/appliance/firewall/vmware-vmcam'
FIREWALL_SCRIPT='/usr/lib/applmgmt/networking/bin/firewall-reload'

PORT_IN_USE = 'cis.error.portinuse'
INVALID_ARGUMENT = 'cis.error.invalidValue'

CAMCONFIG = '/usr/lib/vmware-vmcam/bin/camconfig'
if os.name == 'nt':
    CAMCONFIG = os.path.join(os.environ['VMWARE_CIS_HOME'],
            'vmcamd',
            'camconfig.exe')

class RecoverableRotatingFileHandler(logging.handlers.RotatingFileHandler):
    '''A class that works around a bug in the rotating file handler.'''
    if hasattr(builtins, "WindowsError"):
        def doRollover(self):
            try:
                logging.handlers.RotatingFileHandler.doRollover(self)
            except WindowsError: # pylint: disable=E0602
                # An error can occur if the file handle is open in more than one
                # process, which can happen if we're running a subprocess.
                # For now, we'll just reopen the file to get the object back
                # into a good state.
                self.stream = self._open()

def setupLogger(logDir):
    rootLogger = logging.getLogger()
    if logDir:
        logPath = os.path.join(logDir, 'vmcam-sca.log')
        handler = RecoverableRotatingFileHandler(
                            filename = logPath,
                            mode = "a",
                            maxBytes = LOG_SIZE,
                            backupCount = LOG_ROTATE_COUNT)
        handler.setLevel(logging.DEBUG)
        handler.setFormatter(logging.Formatter(LOG_FORMAT))
        rootLogger.addHandler(handler)
        rootLogger.setLevel(logging.DEBUG)

def run_command(cmd, stdin=None):
    """
    execute a command with the given input and return the return code and output
    """
    rc = stdout = stderr = 0
    try:

        logging.info("Running command: " + str(cmd))

        process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE, stdin=subprocess.PIPE)
        stdout, stderr = process.communicate(stdin.encode())
        rc = process.wait()

        if rc == 0:
            stdout = stdout.rstrip()

        if stderr is not None:
            stderr = stderr.rstrip()
    except:
        logging.exception('Something went wrong while doing %s '
                'rc: %d', str(cmd), rc)

    return rc, stdout, stderr

class VmCamConfig(ServiceConfig):
    '''
    Perform the validations
    for various configuration
    For instance if the port is in use
    Throw Config Error
    saying the port cannot be modified
    to mentioned one
    '''

    def _set(self):
        '''
        No-op since vmcamd updates the properties file during
        add/remove domain operations
        '''
        pass

    def _validate(self):
        '''
        Validates parameters and calls camconfig
        '''
        logging.info('Validating the parameters for vmcam')
        props = _scm_load_properties(sys.stdin.buffer)
        propsfile,_ = self._load_all()

        logging.info('Successfully validated the parameters')
        logging.info('Running camconfig add-domain')
        try:
            # Add AD domain information
            (rc, stdout, stderr) = run_command(
                [CAMCONFIG,
                 'add-domain',
                 '-d', props.get('adDomain', propsfile.get('adDomain')),
                 '-u', props.get('adUser', propsfile.get('adUser'))],
                stdin=props.get('adPassword', ''))
            if rc != 0:
                logging.error("Could not add Active Directory domain")
                logging.error(stdout)
                logging.error(stderr)
                sys.exit(1) # FIXME -- Use real error code
            else:
                logging.info('Active Directory domain added successfully')

        except Exception as e:
            logging.exception('Something went wrong while updating server')

    def _notify(self):
        '''
        When the parameter is being set
        '''
        pass

if os.name.lower() == 'nt':
    configPath = os.path.join(os.environ['VMWARE_DATA_DIR'], 'vmcamd')
    logPath = os.path.join(os.environ['VMWARE_LOG_DIR'], 'vmcamd')

else:
    configPath = '/var/lib/vmware/vmcam'
    logPath = os.path.join(os.environ['VMWARE_LOG_DIR'], 'vmware', 'vmcamd')

propsPath = os.path.join(configPath, 'sca', 'vmcam-config.props')
attrPath = os.path.join(configPath, 'sca', 'vmcam-config.attrs')
'''
If anything goes wrong , it should not be marked as fatal
This is just for some debugging purposes
'''
try:
    setupLogger(logPath)
except:
    pass

VmCamConfig(propsPath, attrPath)
