# Copyright 2014-17 VMware, Inc.  All rights reserved. -- VMware Confidential

import os
import sys
import logging
import socket
import subprocess
import json
import tempfile
import xml.etree.ElementTree as ET
sys.path.append(os.environ['VMWARE_PYTHON_PATH'])
from cis.utils import open_port, close_port

from serviceconfig import ServiceConfig, PLAT_IND_CIS_HOME, ConfigError, \
    _scm_load_properties

dataPath = os.path.join(os.environ['VMWARE_DATA_DIR'], 'netdump')
XMLPath = os.path.join(dataPath, 'netdump-setup.xml')

PORT_IN_USE = 'cis.error.portinuse'
INVALID_ARGUMENT = 'cis.error.invalidValue'

LOG_FORMAT = "%(asctime)s [%(process)d]%(levelname)s:%(module)s:%(message)s"

LOG_SIZE = 10 * 1024 * 1024
LOG_ROTATE_COUNT = 5

FIREWALL_CONFIG = '/etc/vmware/appliance/firewall/vmware-netdumper'
FIREWALL_SCRIPT = '/usr/lib/applmgmt/networking/bin/firewall-reload'

PYTHON_V3 = sys.version_info[0] >= 3


class RecoverableRotatingFileHandler(logging.handlers.RotatingFileHandler):
    '''A class that works around a bug in the rotating file handler.'''
    def doRollover(self):
        try:
            logging.handlers.RotatingFileHandler.doRollover(self)
        except WindowsError:  # pylint: disable=E0602
            # An error can occur if the file handle is open in more than one
            # process, which can happen if we're running a subprocess.
            # For now, we'll just reopen the file to get the object back
            # into a good state.
            self.stream = self._open()


def changeLogFileOwner(logPath, user='deploy'):
    if os.name == 'nt':
        return
    else:
        import pwd
        try:
            if pwd.getpwuid(os.stat(logPath).st_uid).pw_name != user:
                try:
                    pw = pwd.getpwnam(user)
                    current_umask = os.umask(0)
                    os.chown(logPath, pw.pw_uid, pw.pw_gid)
                except:
                    logging.exception('Error while changing ownership: name: %s'
                                      % logPath)
                    raise
        except:
            logging.exception('Unable to find the owner for %s' % logPath)
            raise


def setupLogger(logDir):
    rootLogger = logging.getLogger()
    if logDir:
        logPath = os.path.join(logDir, 'netdump-sca.log')
        if os.path.exists(logPath):
            changeLogFileOwner(logPath)
        handler = RecoverableRotatingFileHandler(
            filename=logPath,
            mode="a",
            maxBytes=LOG_SIZE,
            backupCount=LOG_ROTATE_COUNT)
        handler.setLevel(logging.DEBUG)
        handler.setFormatter(logging.Formatter(LOG_FORMAT))
        rootLogger.addHandler(handler)
        rootLogger.setLevel(logging.DEBUG)


class NetdumpConfig(ServiceConfig):

    def _validate(self):
        logging.info('Validation of netdump parameters started')

        if PYTHON_V3:
            source = sys.stdin.buffer
        else:
            source = sys.stdin
        props = _scm_load_properties(source)
        for key in props:
            if key == 'NETDUMPER_DIR_MAX_GB':
                if not props[key].isdigit():
                    ConfigError(INVALID_ARGUMENT, key,
                                'Please provide integer value for max dir size').fatal()
                if not 1 <= int(props[key]) <= 10:
                    """
                    XXX - what are max/min sizes 1/10 here?
                    10 is the default lvm size for netdumper vmdk
                    to grow outside this size, manual resize of the
                    hdd 9 is needed.
                    """
                    ConfigError(
                        INVALID_ARGUMENT,
                        key,
                        'Please provide max dir size in valid range.').fatal()
            elif key == 'NETDUMPER_PORT':
                if not props[key].isdigit():
                    ConfigError(INVALID_ARGUMENT, key,
                                'Please provide integer value for port').fatal()
                else:
                    serviceport = int(props[key])
                    if serviceport >= 1 and serviceport <= 65535:
                        sock = socket.socket(socket.AF_INET6,
                                             socket.SOCK_STREAM)
                        try:
                            sock.bind(('', serviceport))
                        except Exception:
                            ConfigError(PORT_IN_USE, key,
                                        'The provided port value is already in use').fatal()
                    else:
                        ConfigError(INVALID_ARGUMENT, key,
                                    'Please provide a port number in valid range').fatal()
            logging.info('Successfully validated netdump parameters')

    def _notify(self):
        '''
        Need to do this only for windows as it uses a config xml.
        In linux, netdump works by using the sysconfig file
        On linux and windows we need to update firewall with
        external UDP port.
        '''
        logging.info('Notifying changes to netdump')
        props, attrs = self._load_all()

        if os.name == 'nt':
            try:
                xml = ET.parse(XMLPath)
                defaultValues = xml.getroot().find('defaultValues')

                if defaultValues is not None:
                    for child in defaultValues:
                        if child.tag == 'port':
                            if child.text != props['NETDUMPER_PORT']:
                                close_port(child.text, 'UDP')
                                child.text = props['NETDUMPER_PORT']
                                open_port(child.text, 'UDP',
                                          'VMWare vSphere ESXi Dump '
                                          'Collector')
                        elif child.tag == 'maxSize':
                            child.text = props['NETDUMPER_DIR_MAX_GB']

                with open(XMLPath, 'w') as fp:
                    fp.write(ET.tostring(xml.getroot()))
            except Exception:
                logging.exception(
                    'Something went wrong while updating the XML')

        else:
            with open(FIREWALL_CONFIG, 'r') as fp:
                firewallData = json.load(fp)
                '''
                Expecting file will be in this format,
                else we want exception to be thrown for key error
                '''
                for rule in firewallData['firewall']['rules']:
                    if rule['name'].lower() == 'netdumper.ext.serviceport':
                        rule['port'] = props['NETDUMPER_PORT']
                        logging.info('Firewall data port %s ' % rule['port'])

            with open(FIREWALL_CONFIG, 'w') as fp:
                fp.write(json.dumps(firewallData, indent=4, sort_keys=True))

            obj = subprocess.Popen(FIREWALL_SCRIPT, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE)
            stdout, stderr = obj.communicate()
            logging.info('Reconfigured the firewall service stdout:%s,'
                         ' stderr:%s, returncode:%s' % (stdout, stderr,
                                                        obj.returncode))
        logging.info('Finished notifications')

if os.name == 'nt':
    propsPath = os.path.join(dataPath, 'netdumper.sysconfig')
    logPath = os.path.join(os.environ['VMWARE_LOG_DIR'], 'netdumper')
else:
    propsPath = '/etc/sysconfig/netdumper'
    logPath = os.path.join(os.environ['VMWARE_LOG_DIR'], 'vmware', 'netdumper')

attrPath = os.path.join('%ssca' % PLAT_IND_CIS_HOME,
                        'scripts',
                        'configuration',
                        'netdump-config.attr')

try:
    setupLogger(logPath)
except:
    pass

NetdumpConfig(propsPath, attrPath)
